/***********************************************************************************************************************
* DISCLAIMER
* This software is supplied by Renesas Electronics Corporation and is only 
* intended for use with Renesas products. No other uses are authorized. This 
* software is owned by Renesas Electronics Corporation and is protected under 
* all applicable laws, including copyright laws.
* THIS SOFTWARE IS PROVIDED "AS IS" AND RENESAS MAKES NO WARRANTIES REGARDING 
* THIS SOFTWARE, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING BUT NOT 
* LIMITED TO WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE 
* AND NON-INFRINGEMENT.  ALL SUCH WARRANTIES ARE EXPRESSLY DISCLAIMED.
* TO THE MAXIMUM EXTENT PERMITTED NOT PROHIBITED BY LAW, NEITHER RENESAS 
* ELECTRONICS CORPORATION NOR ANY OF ITS AFFILIATED COMPANIES SHALL BE LIABLE 
* FOR ANY DIRECT, INDIRECT, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES FOR 
* ANY REASON RELATED TO THIS SOFTWARE, EVEN IF RENESAS OR ITS AFFILIATES HAVE 
* BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
* Renesas reserves the right, without notice, to make changes to this software 
* and to discontinue the availability of this software.  By using this software, 
* you agree to the additional terms and conditions found by accessing the 
* following link:
* http://www.renesas.com/disclaimer
*
* Copyright (C) 2010, 2011 Renesas Electronics Corporation. All rights reserved.
***********************************************************************************************************************/

/***********************************************************************************************************************
* File Name    : r_cg_serial_user.c
* Version      : Applilet3 for RL78/G13 V1.03.01 [11 Oct 2011]
* Device(s)    : R5F100GL
* Tool-Chain   : IAR Systems iccrl78
* Description  : This file implements device driver for Serial module.
* Creation Date: 30-03-2012
***********************************************************************************************************************/

/***********************************************************************************************************************
Includes
***********************************************************************************************************************/
#include "r_cg_macrodriver.h"
#include "r_cg_serial.h"
/* Start user code for include. Do not edit comment generated here */
/* End user code. Do not edit comment generated here */
#include "r_cg_userdefine.h"
#include "r_cg_port.h"
#include "r_cg_it.h"
#include "r_cg_intc.h"

#include "r_cg_spi_user.h"

/* API related includes */
#include "hci.h" 
#include "spi.h"

/***********************************************************************************************************************
Global variables and functions
***********************************************************************************************************************/

volatile unsigned char *gp_csi21_rx_address;         /* csi21 receive buffer address */
volatile unsigned short  g_csi21_rx_length;           /* csi21 receive data length */
volatile unsigned short  g_csi21_rx_count;            /* csi21 receive data count */
volatile unsigned char *gp_csi21_tx_address;         /* csi21 send buffer address */
volatile unsigned short  g_csi21_send_length;         /* csi21 send data length */
volatile unsigned short  g_csi21_tx_count;            /* csi21 send data count */

/* Start user code for global. Do not edit comment generated here */

/* SPI Related Defines */
static volatile unsigned char gCallSpiIntHandler = 0;
uint8_t wlan_rx_buffer[SPI_RX_BUFFER_SIZE];
uint8_t wlan_tx_buffer[SPI_TX_BUFFER_SIZE];

void csi21_UserInit(void);
void csi21_InterruptDisable(void);

static void SPIWaitForTxFinish(void);
void SpiWriteDataSynchronous(uint8_t *data, unsigned short size); 
void CC3000_SendReceiveData(unsigned char *txbuf, unsigned short txnum, unsigned char *rxbuf);
void SpiReadWriteString(uint8_t ulTrueFalse, const uint8_t *ptrData, uint16_t ulDataSize);
void SpiReadWriteStringInt(uint8_t ulTrueFalse, const uint8_t *ptrData, uint16_t ulDataSize);
void SpiIntGPIOHandler(void);

#define 	eSPI_STATE_POWERUP 			(0)
#define 	eSPI_STATE_INITIALIZED  		(1)
#define 	eSPI_STATE_IDLE			        (2)
#define 	eSPI_STATE_WRITE_IRQ	   		(3)
#define 	eSPI_STATE_WRITE_FIRST_PORTION          (4)
#define 	eSPI_STATE_WRITE_EOT			(5)
#define 	eSPI_STATE_READ_IRQ		        (6)
#define 	eSPI_STATE_READ_FIRST_PORTION	        (7)
#define 	eSPI_STATE_READ_EOT			(8)

#define ASSERT_CS()          { SPI_CS_PIN = FALSE;}
#define DEASSERT_CS()        { SPI_CS_PIN = TRUE; }

typedef struct
{
	gcSpiHandleRx  SPIRxHandler;
	unsigned short usTxPacketLength;
	unsigned short usRxPacketLength;
	volatile unsigned long  ulSpiState;
	unsigned char *pTxPacket;
	unsigned char *pRxPacket;
	
}tSpiInformation;


#pragma pack(1)
typedef struct _btspi_hdr
{
    unsigned char   cmd;
    unsigned short  length;
    unsigned char   pad[2];
}btspi_hdr;

tSpiInformation sSpiInformation;

// Static buffer for 5 bytes of SPI HEADER
uint8_t tSpiReadHeader[] = {READ, 0,0, 0, 0};

/* This buffer is used when we are sending data on SPI but we are not interested
in reading from Slave*/
unsigned char chBuffer[32]; 
/***********************************************************************************************************************
* Function Name: R_SAU1_Create
* Description  : This function initializes the SAU1 module.
* Arguments    : None
* Return Value : None
***********************************************************************************************************************/
void R_SAU1_Create(void)
{
    SAU1EN = 1U;    /* supply SAU1 clock */
    NOP();
    NOP();
    NOP();
    NOP();
    SPS1 = _0000_SAU_CK00_FCLK_0 | _0000_SAU_CK01_FCLK_0; // TBK: 32 MHz 
    R_CSI21_Create();
}

/***********************************************************************************************************************
* Function Name: R_CSI21_Create
* Description  : This function initializes the CSI21 module.
* Arguments    : None
* Return Value : None
***********************************************************************************************************************/
void R_CSI21_Create(void)
{
    ST1 |= _0002_SAU_CH1_STOP_TRG_ON;    /* disable CSI21 */
    CSIMK21 = 1U;    /* disable INTCSI21 interrupt */
    CSIIF21 = 0U;    /* clear INTCSI21 interrupt flag */
    /* Set INTCSI21 low priority */
    CSIPR121 = 0U;
    CSIPR021 = 0U;
    SIR11 = _0004_SAU_SIRMN_FECTMN | _0002_SAU_SIRMN_PECTMN | _0001_SAU_SIRMN_OVCTMN;    /* clear error flag */
    SMR11 = _0020_SAU_SMRMN_INITIALVALUE | _0000_SAU_CLOCK_SELECT_CK00 | _0000_SAU_CLOCK_MODE_CKS |
            _0000_SAU_TRIGGER_SOFTWARE | _0000_SAU_MODE_CSI | _0000_SAU_TRANSFER_END;
    SCR11 = _C000_SAU_RECEPTION_TRANSMISSION | _1000_SAU_TIMING_2 | _0000_SAU_MSB | _0007_SAU_LENGTH_8;
    
    SDR11 = _8MHz_SPI_CLK_DIVISOR;
    
   // SDR11 = _16MHz_SPI_CLK_DIVISOR;
    
   // SDR11 = _32MHz_SPI_CLK_DIVISOR;
    
    //SDR11 = _1MHz_SPI_CLK_DIVISOR;
  
    
    SO1 &= ~_0200_SAU_CH1_CLOCK_OUTPUT_1;    /* CSI21 clock initial level */
    SO1 &= ~_0002_SAU_CH1_DATA_OUTPUT_1;    /* CSI21 SO initial level */
    SOE1 |= _0002_SAU_CH1_OUTPUT_ENABLE;    /* enable CSI21 output */
    /* Set SI21 pin */
    PM7 |= 0x02U;
    /* Set SO21 pin */
    P7 |= 0x04U;
    PM7 &= 0xFBU;
    /* Set SCK21 pin */
    P7 |= 0x01U;
    PM7 &= 0xFEU;
}

/***********************************************************************************************************************
* Function Name: R_CSI21_Start
* Description  : This function starts the CSI21 module operation.
* Arguments    : None
* Return Value : None
***********************************************************************************************************************/
void R_CSI21_Start(void)
{
    CSIIF21 = 0U;    /* clear INTCSI21 interrupt flag */
    CSIMK21 = 1U;    /* enable INTCSI21    <--  WHY DISABLE    */
    SO1 &= ~_0200_SAU_CH1_CLOCK_OUTPUT_1;       /* CSI21 clock initial level */
    SO1 &= ~_0000_SAU_CH1_DATA_OUTPUT_0;           /* CSI21 SO initial level */
    SOE1 |= _0002_SAU_CH1_OUTPUT_ENABLE;           /* enable CSI21 output */
    SS1 |= _0002_SAU_CH1_START_TRG_ON;             /* enable CSI21 */
    csi21_UserInit();
}

/***********************************************************************************************************************
* Function Name: R_CSI21_Stop
* Description  : This function stops the CSI21 module operation.
* Arguments    : None
* Return Value : None
***********************************************************************************************************************/
void R_CSI21_Stop(void)
{
    ST1 |= _0002_SAU_CH1_STOP_TRG_ON;        /* disable CSI21 */
    SOE1 &= ~_0002_SAU_CH1_OUTPUT_ENABLE;    /* disable CSI21 output */
    CSIMK21 = 1U;    /* disable INTCSI21 interrupt */
    CSIIF21 = 0U;    /* clear INTCSI21 interrupt flag */
}

/***********************************************************************************************************************
* Function Name: R_CSI21_Send_Receive
* Description  : This function sends and receives CSI21 data.
* Arguments    : tx_buf -
*                    transfer buffer pointer
*                tx_num -
*                    buffer size
*                rx_buf -
*                    receive buffer pointer
* Return Value : status -
*                    MD_OK or MD_ARGERROR
***********************************************************************************************************************/
MD_STATUS R_CSI21_Send_Receive(uint8_t * const tx_buf, uint16_t tx_num, uint8_t * const rx_buf)
{
    MD_STATUS status = MD_OK;

    if (tx_num < 1U)
    {
        status = MD_ARGERROR;
    }
    else
    {
        g_csi21_tx_count = tx_num;        /* send data count */
        gp_csi21_tx_address = tx_buf;     /* send buffer pointer */
        gp_csi21_rx_address = rx_buf;     /* receive buffer pointer */
        CSIMK21 = 1U;                     /* disable INTCSI21 interrupt */
        SIO21 = *gp_csi21_tx_address;    /* started by writing data to SDR[7:0] */
        gp_csi21_tx_address++;
        g_csi21_tx_count--;
        CSIMK21 = 0U;                     /* enable INTCSI21 interrupt */
    }

    return (status);
}

/* Start user code for adding. Do not edit comment generated here */
void csi21_InterruptDisable(void)
{
	R_CSI21_Stop();      
        SOE1 |= _0002_SAU_CH1_OUTPUT_ENABLE;           /* enable CSI21 output */
        SS1 |= _0002_SAU_CH1_START_TRG_ON;             /* enable CSI21 */
}

void csi21_UserInit(void)
{     
	/* Set MISO for input to TTL*/
	PM7 =_02_PIMn1_TTL_ON;
	/* Set IRQ for input to TTL*/
	PM7 = _08_PIMn3_TTL_ON;   //  _01_PIMn0_TTL_ON;
}

/***********************************************************************************************************************
* Function Name: r_csi21_interrupt
* Description  : This function is INTCSI21 interrupt service routine.
* Arguments    : None
* Return Value : None
***********************************************************************************************************************/
#pragma vector = INTCSI21_vect
__interrupt static void r_csi21_interrupt(void)
{
    unsigned char err_type;

    err_type = (unsigned char)(SSR11 & _0001_SAU_OVERRUN_ERROR);
    SIR11 = (unsigned short)err_type;

    if (1U == err_type)
    {
        r_csi21_callback_error(err_type);    /* overrun error occurs */
    }
    else
    {
        if (g_csi21_tx_count > 0U)
        {
            *gp_csi21_rx_address = SIO21;
            gp_csi21_rx_address++;
            SIO21 = *gp_csi21_tx_address;
            gp_csi21_tx_address++;
            g_csi21_tx_count--;
        }
        else 
        {
            if (0U == g_csi21_tx_count)
            {
                *gp_csi21_rx_address = SIO21;
            }

          //r_csi21_callback_sendend();    /* complete send */
            r_csi21_callback_receiveend();    /* complete receive */
        }
    }
}
/*
**-----------------------------------------------------------------------------
**
**  Abstract:
**	This function sends and receives CSI21 data.
**
**  Parameters:
**	txbuf: transfer buffer pointer
**	txnum: buffer size
**	rxbuf: receive buffer pointer
**
**  Returns:
**	MD_OK
**	MD_ARGERROR
**
**-----------------------------------------------------------------------------
*/
void CC3000_SendReceiveData(unsigned char *txbuf, unsigned short txnum, unsigned char *rxbuf)
{
		g_csi21_tx_count = txnum;		/* send data count */
		gp_csi21_tx_address = txbuf;	/* send buffer pointer */
		gp_csi21_rx_address = rxbuf;	/* receive buffer pointer */
		while(g_csi21_tx_count)
		{
			SIO21 = *gp_csi21_tx_address;	/* started by writing data to SDR[7:0] */
			gp_csi21_tx_address++;
			g_csi21_tx_count--;
			/* Wait until end of TX */
			SPIWaitForTxFinish();
			*gp_csi21_rx_address = SIO21;
			gp_csi21_rx_address++;
		}
		SysCtlDelay(1000);
}

/***********************************************************************************************************************
* Function Name: r_csi21_callback_receiveend
* Description  : This function is a callback function when CSI21 finishes reception.
* Arguments    : None
* Return Value : None
***********************************************************************************************************************/
void r_csi21_callback_receiveend(void)
{
   /* Start user code. Do not edit comment generated here */
	if(gCallSpiIntHandler)
	{
		/* We are interested in intering the main SPI Interrupt handling routine*/
		//SpiIntHandler();
	}
	/* End user code. Do not edit comment generated here */
}

/***********************************************************************************************************************
* Function Name: r_csi21_callback_error
* Description  : This function is a callback function when CSI21 reception error occurs.
* Arguments    : err_type -
*                    error type value
* Return Value : None
***********************************************************************************************************************/
void r_csi21_callback_error(uint8_t err_type)
{
    /* Start user code. Do not edit comment generated here */
    /* End user code. Do not edit comment generated here */
}

/***********************************************************************************************************************
* Function Name: r_csi21_callback_sendend
* Description  : This function is a callback function when CSI21 finishes transmission.
* Arguments    : None
* Return Value : None
***********************************************************************************************************************/
void r_csi21_callback_sendend(void)
{
    /* Start user code. Do not edit comment generated here */
    /* End user code. Do not edit comment generated here */
}

/* Start user code for adding. Do not edit comment generated here */


/**
  * @brief  The functions delay for a number of MCU clk period
  * @param  The number of MCU clk period to delay for
  * @retval None
  */

inline void SysCtlDelay(unsigned long Delay)
{
	/* Decrement nCount value */
  unsigned long ulDelay = Delay;	
  while (ulDelay != 0)
  {
    ulDelay--;
  } 
}

/**
  * @brief  This functions delays for one second
  * @param  None
  * @retval None
  */

void SpiDelayOneSecond(void)
{
	unsigned long ulIndex =10000;
	/*Delay for 1 second*/
	while (ulIndex != 0 )
	{
		SysCtlDelay(FIFTY_US_DELAY);
		SysCtlDelay(FIFTY_US_DELAY);
		ulIndex--;
	}
}

/**
  * @brief  Enables or disables SPI Interrupt
  * @param  None 
  * @retval None
  */
void SpiInterruptDisable(void)
{ 
	gCallSpiIntHandler = FALSE;	
}
/**
  * @brief  Ensure the main SPI Int handler is called after all SPI data is sent
  * @param  None
  * @retval None
  */
void SpiInterruptEnable(void)
{ 
	gCallSpiIntHandler = TRUE;
}

/**
  * @brief  Clears SPI interrupt flags
  * @param  None
  * @retval None
  */
void SpiClearInterruptFlag(void)
{
  
        CSIIF21= 0U;	/* clear INTCSI21 interrupt flag */
}


/**
  * @brief  Enables Interrupt Detection on IRQ input pin
  * @param  None
  * @retval None
  */
void WlanInterruptEnable(void)
{
        R_INTC5_Start();
}
/**
  * @brief  Disables Interrupt Detection on IRQ input pin
  * @param  None
  * @retval None
  */
void WlanInterruptDisable(void)
{
        R_INTC5_Stop();     
}

/**
  * @brief  Reads the status of the IRQ pin from CC3000
  * @param  None
  * @retval TRUE or FALSE
  */
long ReadWlanInterruptPin()
{
	return SPI_IRQ_PIN;
}

/** @brief  Opens the Spi Port and specify and ISR Handler
  * @param  Data: byte to send.
  * @retval None
  */
void SpiOpen(gcSpiHandleRx pfRxHandler)
{
	sSpiInformation.ulSpiState = eSPI_STATE_POWERUP;
	sSpiInformation.SPIRxHandler = pfRxHandler;
	sSpiInformation.pTxPacket = wlan_tx_buffer;
	sSpiInformation.pRxPacket = wlan_rx_buffer;
	sSpiInformation.usRxPacketLength = 0;
	sSpiInformation.usTxPacketLength = 0;     
        
        tSLInformation.WlanInterruptEnable(); 	
}

/**
  * @brief  Opens the Spi Port and specify and ISR Handler
  * @param  Data: byte to send.
  * @retval None
  */
void SpiClose(void)
{
	if (sSpiInformation.pRxPacket)
	{
		sSpiInformation.pRxPacket = 0;
	}
	//
	//	Disable Interrupt in GPIOA module...
	//
    WlanInterruptDisable();
}

/**
  * @brief  Sends data on SPI to generate interrupt on reception
  * @param  The pointer to data buffer
  * @param  This size of data
  * @retval None
  */
void SpiReadData(unsigned char *data, unsigned short size)
{
	SpiReadWriteStringInt(TRUE,data,size);
}

/**
  * @brief  A wrapper to enable interrupt on IRQ line
  * @param  None
  * @retval None
  */
void SpiResumeSpi(void)
{
	//
	//	Enable Interrupt on WLAN IRQ Pin
	//
    WlanInterruptEnable();	
}

void SpiPauseSpi(void)
{
	//
	//	Enable Interrupt on WLAN IRQ Pin
	//
    WlanInterruptDisable();	
}
/**
  * @brief  This indicate the end of a receive and calls a registered handler 
			to process the received data
  * @param  None
  * @retval None
  */
void SpiTriggerRxProcessing(void)
{
	//
	// Trigger Rx processing
	//
	SpiPauseSpi();
	DEASSERT_CS();
	
	sSpiInformation.ulSpiState = eSPI_STATE_IDLE;
	sSpiInformation.SPIRxHandler(sSpiInformation.pRxPacket + sizeof(btspi_hdr));
}
/**
  * @brief  This sends data over the SPI transport layer with
  * @param  None
  * @retval None
  */
void SpiWriteAsync(const unsigned char *data, unsigned short size)
{
	SpiReadWriteString(FALSE, data, size);	
}
/**
  * @brief  Ensures that the CS is not asserted before the SPI finish 
			transmitting data
  * @param  None
  * @retval None
  */
static void SPIWaitForTxFinish()
{
    /* Loop until the end of SPI transmit */
	while(!( SSR11 &_0020_SAU_VALID_STORED))	
	{
        ;
	}
}	

/**
  * @brief  This function TX and RX SPI data and configures interrrupt generation
  * 		at the end of the TX interrupt
  * @param  ulTrueFlase True for a read or False for write
  * @param  ptrData Pointer to data to be written 
  * @param  ulDataSize The size of the data to be written or read
  * @retval None
  */
void SpiReadWriteStringInt(uint8_t ulTrueFalse, const uint8_t *ptrData, uint16_t ulDataSize)
{
	/* Wait for the end of SPI Transport Layer Transaction*/
	SpiInterruptEnable();	
        SysCtlDelay(FIFTY_US_DELAY);	
	if(ulTrueFalse == TRUE)
	{
		CC3000_SendReceiveData((uint8_t*)tSpiReadHeader, ulDataSize,(uint8_t*)ptrData );			
	}
	else
	{
		CC3000_SendReceiveData((uint8_t*)ptrData, ulDataSize,(uint8_t*)sSpiInformation.pRxPacket );				
	}	
	/* Wait for the end of SPI Transport Layer Transaction*/
	//SPIWaitForTxFinish();	
	//SysCtlDelay(FIFTY_US_DELAY);	
	r_csi21_callback_receiveend();		
}

/**
  * @brief  This function TX and RX SPI data with no interrupt at end of SPI TX
  * @param  ulTrueFlase True for a read or False for write
  * @param  ptrData Pointer to data to be written 
  * @param  ulDataSize The size of the data to be written or read
  * @retval None
  */
void SpiReadWriteString(uint8_t ulTrueFalse, const uint8_t *ptrData, uint16_t ulDataSize)
{
	SpiInterruptDisable();	
        SysCtlDelay(FIFTY_US_DELAY);	
	if(ulTrueFalse == TRUE)
	{
		CC3000_SendReceiveData((uint8_t*)tSpiReadHeader, ulDataSize,(uint8_t*)ptrData );			
	}
	else
	{
		CC3000_SendReceiveData((uint8_t*)ptrData, ulDataSize,(uint8_t*)sSpiInformation.pRxPacket );				
	}			
}

/**
  * @brief  This function TX and RX SPI data with no interrupt at end of SPI TX
  * @param  ptrData Pointer to data to be written 
  * @param  ulDataSize The size of the data to be written or read
  * @retval None
  */
void SpiWriteDataSynchronous(uint8_t *data, unsigned short size)
{
	SpiReadWriteString(TRUE, data, size);
}
/**
  * @brief  Sends header information to CC3000
  * @param  None
  * @retval None
  */
static long SpiFirstWrite(unsigned char *ucBuf, unsigned short usLength)
{	
    //
    // workaround for first transaction
    //
    ASSERT_CS();
    //SPI writes first 4 bytes of data
    SpiReadWriteString(FALSE, ucBuf, 4);
    //SPI writes next 4 bytes of data	
    SpiReadWriteString(FALSE, ucBuf + 4, usLength - 4);
    // From this point on - operate in a regular way
    sSpiInformation.ulSpiState = eSPI_STATE_IDLE;	
    DEASSERT_CS();
    return(0);
}

/**
  * @brief  Writes data over SPI  transport link to CC3000
  * @param  pUserBuffer: pointer to data 
  * @param usLength: length of data that will be sent to CC3000
  * @retval None
  */
long SpiWrite(unsigned char *pUserBuffer, unsigned short usLength)
{
    unsigned char ucPad = 0;
	//
	// Figure out the total length of the packet in order to figure out if there is padding or not
	//
    if(!(usLength & 0x0001))
    {
        ucPad++;
    }

    pUserBuffer[0] = WRITE;
    pUserBuffer[1] = HI(usLength + ucPad);
    pUserBuffer[2] = LO(usLength + ucPad);
    pUserBuffer[3] = 0;
    pUserBuffer[4] = 0;

    usLength += (sizeof(btspi_hdr) + ucPad);
	
	if (sSpiInformation.ulSpiState == eSPI_STATE_POWERUP)
	{
		while (sSpiInformation.ulSpiState != eSPI_STATE_INITIALIZED);
	}
	if (sSpiInformation.ulSpiState == eSPI_STATE_INITIALIZED)
	{	
		// This is time for first TX/RX transactions over SPI: the IRQ is down - so need to send read buffer size command
		SpiFirstWrite(pUserBuffer, usLength);
	}
	else  
	{
		// We need to prevent here race that can occur in case 2 back to back packets are sent to the 
		// device, so the state will move to IDLE and once again to not IDLE due to IRQ
		//tSLInformation.WlanInterruptDisable();
		while (sSpiInformation.ulSpiState != eSPI_STATE_IDLE)
		{
			;
		}
		sSpiInformation.ulSpiState = eSPI_STATE_WRITE_IRQ;
		sSpiInformation.pTxPacket = pUserBuffer;
		sSpiInformation.usTxPacketLength = usLength;
		//
		// Assert the CS line and wait till SSI IRQ line is active and then initialize write operation
		//
		//tSLInformation.WlanInterruptEnable(); 		
		ASSERT_CS();
		
	}
	// Due to the fact that we are currently implementing a blocking situation
	// here we will wait till end of transaction
	//	
	while (eSPI_STATE_IDLE != sSpiInformation.ulSpiState)
	{
		;
	}  	
    return(0);
}

/**
  * @brief  This function processes received SPI Header and in accordance with it
		- continues reading the packet 		
  * @param  None
  * @retval None
  */
long SpiReadDataCont(void)
{
    hci_hdr_t *hci_hdr;
    long data_to_recv;
	hci_evnt_hdr_t *hci_evnt_hdr;
	unsigned char *evnt_buff;
	hci_data_hdr_t *pDataHdr;
    //
    //determine what type of packet we have
    //
    evnt_buff =  sSpiInformation.pRxPacket;
    data_to_recv = 0;
	hci_hdr = (hci_hdr_t *)(evnt_buff + sizeof(btspi_hdr));
	
    switch(hci_hdr->ucType)
    {
        case HCI_TYPE_DATA:
        {
			pDataHdr = (hci_data_hdr_t *)(evnt_buff + sizeof(btspi_hdr));

			//
			// We need to read the rest of data..
			//
			data_to_recv = pDataHdr->usLength;

			if (!((HEADERS_SIZE_EVNT + data_to_recv) & 1))
			{	
    	        data_to_recv++;
			}

			if (data_to_recv)
			{
            	SpiReadWriteString(TRUE,evnt_buff + 10, data_to_recv);
			}
            break;
        }
        case HCI_TYPE_EVNT:
        {
        	//
            //configure buffer to read rest of the data
            //
            hci_evnt_hdr = (hci_evnt_hdr_t *)hci_hdr;

			// 
			// Calculate the rest length of the data
			//
                        data_to_recv = hci_evnt_hdr->ucLength - 1;
			
			// 
			// Add padding byte if needed
			//
			if ((HEADERS_SIZE_EVNT + data_to_recv) & 1)
			{
				
	            data_to_recv++;
			}
			
			if (data_to_recv)
			{
            	          SpiReadWriteString(TRUE, evnt_buff + 10, data_to_recv);
			}

			sSpiInformation.ulSpiState = eSPI_STATE_READ_EOT;
            break;
        }
    }
	
    return (0);
}


/**
  * @brief  This function enter point for read flow: first we read minimal 
			5 SPI header bytes and 5 Event Data bytes 
			
  * @param  None
  * @retval None
  */
void SpiReadHeader(void)
{
	sSpiInformation.ulSpiState = eSPI_STATE_READ_IRQ;
	SpiReadWriteString(TRUE, sSpiInformation.pRxPacket, 10);
}

/**
  * @brief  Determine if all data was read if so end the data exchange
  * @param  None
  * @retval None
  */
void SSIContReadOperation(void)
{
	//
	// The header was read - continue with  the payload read
	//
	if (!SpiReadDataCont())
	{
		//
		// All the data was read - finalize handling by switching to teh task
		//	and calling from task Event Handler
		//
		SpiTriggerRxProcessing();
	}
}

/**
  * @brief  The handler for Interrupt that is generated when CC3000 brings the 
			IRQ line low.
  * @param  None
  * @retval None
  */
void SpiIntGPIOHandler(void)
{
	//Flag is cleared in first ISR handler

    if(!ReadWlanInterruptPin())
    {

        if (sSpiInformation.ulSpiState == eSPI_STATE_POWERUP)
        {
            /* This means IRQ line was low call a callback of HCI Layer to inform on event */
            sSpiInformation.ulSpiState = eSPI_STATE_INITIALIZED;
        }
        else if (sSpiInformation.ulSpiState == eSPI_STATE_IDLE)
        {
			//SPIWaitForTxFinish();
			//WDT_Restart();
			sSpiInformation.ulSpiState = eSPI_STATE_READ_IRQ;
            /* IRQ line goes down - we are starting reception */
            ASSERT_CS();    
            //
            // Wait for TX/RX Complete which will come as DMA interrupt
            // 
            SpiReadHeader();
			sSpiInformation.ulSpiState = eSPI_STATE_READ_EOT;
			
			//
			//
			//
			SSIContReadOperation();			
        }
        else if (sSpiInformation.ulSpiState == eSPI_STATE_WRITE_IRQ)
        {
			
			SpiReadWriteString(FALSE,sSpiInformation.pTxPacket, sSpiInformation.usTxPacketLength);
			sSpiInformation.ulSpiState = eSPI_STATE_IDLE;
			DEASSERT_CS();
			
        }
    }	
}

